/* 
 * Seal.o - seal the kernel so no further modules may be loaded.  
 * Also disables /dev/kmem
 * gcc -O6 -c seal.c -I/usr/src/linux/include -DMODULE -D__KERNEL__ -Wall
 * jcl@cmu.edu & dmarg@cs.cmu.edu
 * Released under GPL
 *
 * Arguments: "[read=not] [insmod=/sbin/insmod] 
    [mem_allow=/usr/X11R6/bin/XF86_SVGA] [no_remove=true]"
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/smp_lock.h>
#include <linux/mm.h>

char *read=NULL;
MODULE_PARM(read, "s");
char *insmod=NULL;
MODULE_PARM(insmod, "s");
char *mem_allow=NULL;
MODULE_PARM(mem_allow, "s");
char *no_remove=NULL;
MODULE_PARM(no_remove, "s");

int insmod_inode;
kdev_t insmod_dev;

int mem_allow_inode;
kdev_t mem_allow_dev;

typedef int (*sysfun_p)();
extern sysfun_p sys_call_table[];

#define DEBUG 0

sysfun_p real_sys_create_module;
sysfun_p real_sys_init_module;
sysfun_p real_sys_open;

struct md5_name {
  char *modname;
  char md5sum[16];
} *md5map;

int no_create_module(const char *name_user, size_t size)
{     
  printk(KERN_DEBUG "seal: Attempt to create module.\n");

  if (
      insmod && current && current->mm && current->mm->mmap 
      && current->mm->mmap->vm_file && current->mm->mmap->vm_file->f_dentry
      && current->mm->mmap->vm_file->f_dentry->d_inode
      && current->mm->mmap->vm_file->f_dentry->d_inode->i_ino == insmod_inode
      && current->mm->mmap->vm_file->f_dentry->d_inode->i_sb
      && current->mm->mmap->vm_file->f_dentry->d_inode->i_sb->s_dev==insmod_dev
      ) 
    return real_sys_create_module(name_user, size);
  else 
    return -EPERM; 
}

int no_init_module(const char *name_user, struct module *mod_user)
{ 
  printk(KERN_DEBUG "seal: Attempt to init module.\n");

  if ( 
      insmod && current && current->mm && current->mm->mmap 
      && current->mm->mmap->vm_file && current->mm->mmap->vm_file->f_dentry
      && current->mm->mmap->vm_file->f_dentry->d_inode
      && current->mm->mmap->vm_file->f_dentry->d_inode->i_ino == insmod_inode
      && current->mm->mmap->vm_file->f_dentry->d_inode->i_sb
      && current->mm->mmap->vm_file->f_dentry->d_inode->i_sb->s_dev==insmod_dev
      ) 
    return real_sys_init_module(name_user, mod_user);
  else 
    return -EPERM; 
}

int no_remove_module(const char *name_user)
{ 
  printk(KERN_DEBUG "seal: Attempt to delete module.\n");
  return -EPERM; 
}

int careful_sys_open(const char * filename, int flags, int mode)
{
  struct dentry *dopen;
  int error = 0;

  lock_kernel();
  dopen = __namei(filename,flags);
  if (IS_ERR(dopen) )
    goto out;

  if ( ! ( dopen->d_inode && current && current->mm 
	    && current->mm->mmap 
	    && current->mm->mmap->vm_file 
	    && current->mm->mmap->vm_file->f_dentry
	    && current->mm->mmap->vm_file->f_dentry->d_inode
	    && current->mm->mmap->vm_file->f_dentry->d_inode->i_sb ) ) {
    printk(KERN_DEBUG "seal: Faulting current = %p, dopen = %p, name = %s\n", current, dopen, filename);
    error = -EFAULT;
  }
  
  if ( /* if major number = 1 and minor = 1,2 then fail. */
      MAJOR(dopen->d_inode->i_rdev) == 1 
      && ( MINOR(dopen->d_inode->i_rdev) == 1 || MINOR(dopen->d_inode->i_rdev) == 2 ) 
      && ( !mem_allow 
	   || current->mm->mmap->vm_file->f_dentry->d_inode->i_ino != mem_allow_inode
	   || current->mm->mmap->vm_file->f_dentry->d_inode->i_sb->s_dev != mem_allow_dev) 
      && (read || flags & O_WRONLY || flags & O_RDWR) ) {
    printk(KERN_DEBUG "seal: Denied major = %i, minor = %i\n", MAJOR(dopen->d_inode->i_rdev), MINOR(dopen->d_inode->i_rdev));
    error = -EPERM;
  }
  
  dput(dopen); 
 out:
  unlock_kernel();

  if ( IS_ERR(error) )
    return error;
  else 
    return real_sys_open(filename, flags, mode);
}

int init_module() {
  struct dentry *dinsmod, *dmem_allow;

  printk(KERN_DEBUG "seal: In init_modules.\n");

  lock_kernel();
  if ( insmod ) {
    dinsmod = open_namei(insmod,0,0);
    if (IS_ERR(dinsmod) || !dinsmod->d_inode || !dinsmod->d_inode->i_sb
	|| !dinsmod->d_inode->i_sb->s_dev) {
      printk(KERN_DEBUG "seal: Can not open /dev/mem or /dev/kmem\n");
      return -EFAULT;      
    }
    insmod_inode = dinsmod->d_inode->i_ino;
    insmod_dev = dinsmod->d_inode->i_sb->s_dev;
  }
  if ( mem_allow ) {
    dmem_allow = open_namei(mem_allow,0,0);
    if (IS_ERR(dmem_allow) || !dmem_allow->d_inode || !dmem_allow->d_inode->i_sb
	|| !dmem_allow->d_inode->i_sb->s_dev) {
      printk(KERN_DEBUG "seal: Can not open /dev/mem or /dev/kmem\n");
      return -EFAULT;      
    }
    mem_allow_inode = dmem_allow->d_inode->i_ino;
    mem_allow_dev = dmem_allow->d_inode->i_sb->s_dev;
  }

  real_sys_open = sys_call_table[5];
  real_sys_create_module = sys_call_table[127];
  real_sys_init_module = sys_call_table[128];
  
  sys_call_table[5] = careful_sys_open;
  sys_call_table[127] = no_create_module;
  sys_call_table[128] = no_init_module;
  if (no_remove || !DEBUG)
    sys_call_table[129] = no_remove_module;
  unlock_kernel();
  
  return 0;
}

void cleanup_module() {

  printk(KERN_DEBUG "seal: In cleanup_module.\n");
  if (DEBUG) {
    lock_kernel();
    sys_call_table[127]=real_sys_create_module;
    sys_call_table[128]=real_sys_init_module;
    sys_call_table[5] = real_sys_open;
    unlock_kernel();
  }
}
